'use client';
import React, { useState, useEffect, useRef } from 'react';
import styles from "../../assets/style.module.css";
import { useSession } from "next-auth/react";
import Loading from '../loading';
import Swal from 'sweetalert2';
import { signOut } from 'next-auth/react';
import { PieChart, Pie, Cell, Tooltip, Legend } from "recharts";

const COLORS = ['#07b0c6', '#3a53a3', '#26a7df'];
const top3SectorColor = ['#07b0c6', '#3a53a3', '#26a7df'];

const SectorChart = () => {
    const [investorPoftfolio, setInvestorPortfolio] = useState([]);
    const [isLoading, setIsLoading] = useState(false);
    const { data } = useSession();
    const prevToken = useRef(null);

    useEffect(() => {
        let token;
        if (data?.user?.token && data.user.token !== prevToken.current) {
            token = data.user.token;
            prevToken.current = token;
            setIsLoading(true);
            fetchPortfolioData(token);
        }
    }, [data?.user?.token]);

    const fetchPortfolioData = async (token) => {
        try {
            const url = process.env.NEXT_PUBLIC_API_URL + '/dashboard-sector-info';
            const res = await fetch(url, {
                method: "GET",
                headers: {
                    "Accept": "application/json",
                    "Content-Type": "application/json",
                    "Authorization": `Bearer ${token}`,
                },
            });
            const responseData = await res.json();
            if (responseData.status === 'success') {
                setInvestorPortfolio(responseData.top_sector);
            } else if (res.status === 401) {
                Swal.fire({
                    position: "top-end",
                    icon: "warning",
                    title: responseData.message,
                    showConfirmButton: false,
                    timer: 1000,
                }).then(() => {
                    signOut();
                });
            }
        } catch (error) {
            Swal.fire({
                position: "top-end",
                icon: "error",
                title: 'The server is not responding. Please wait for a moment.',
                showConfirmButton: false,
                timer: 2000,
            });
        } finally {
            setIsLoading(false);
        }
    };

    if (isLoading) {
        return <Loading loading={isLoading} />;
    }

    const chartData = investorPoftfolio.map((item) => ({
        name: item.sector_name,
        value: parseFloat(item.total_cost_amount),
    }));

    return (
        <div className={`${styles.d_flex} ${styles.w_100} ${styles.card} ${styles.mx_10} `}>
            {investorPoftfolio.length > 0 ? (
                <div
                style={{
                  display: "flex",
                  justifyContent: "center",
                  alignItems: "center",
                  height: "80vh", // Ensures full height for centering
                  flexDirection: "column", // Places title and chart in a column
                }}
              >
                <h3 className={`${styles.gradient_text_green} ${styles.w_100} ${styles.text_center}`}  >
                Top 3 Sectors
                </h3>
                <PieChart width={500} height={500}>
                  <Pie
                    data={chartData}
                    dataKey="value"
                    nameKey="name"
                    cx="50%" // Centers horizontally
                    cy="50%" // Centers vertically
                    outerRadius={150}
                    fill="#8884d8"
                    label
                  >
                    {chartData.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                    ))}
                  </Pie>
                  <Tooltip />
                 <Legend
                        verticalAlign="top"
                        wrapperStyle={{
                            textAlign: "center", // Centers the legend text 
                        }}
                    />
                </PieChart>
              </div> 
            ) : (
                <h2>No data available for the Sector</h2>
            )}
        </div>
    );
};

export default SectorChart;
